;**********************************************************************
;   Copyright (C) 2005 Dwayne Forsyth
;                                 
;   This program is free software; you can redistribute it and/or
;   modify it under the terms of the GNU General Public License
;   as published by the Free Software Foundation; either version 2
;   of the License, or (at your option) any later version.
; 
;   This program is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this program; if not, write to the
; 
;      Free Software Foundation, Inc.
;      51 Franklin Street, Fifth Floor
;      Boston, MA  02110-1301, USA.
;
;**********************************************************************
;
;   This software drives a 35-LED circuit board also designed by Dwayne
;   Forsyth. That flash the bi-color LEDs in pretty patterns and can
;   scroll messages on a 5x7 bi-color LED matrix. The 1st introduction
;   of this blinkie kit is to be Duckon 15 (www.duckon.com) and it will
;   be sold on the www.2dkits.com webpage.
;
;   If you use this code, find a bug, or improve it please sent an email
;   to Dwayne.                                                    
;                                                                     
;**********************************************************************
;                                                                     
;    Filename:	    5x7.asm                                          
;    Date:          05/11/2006                                        
;    File Version:  V3                                                
;                                                                     
;    Author:  Dwayne Forsyth
;    email:   Dwayne@2DKits.com                                         
;    Company: www.2DKits.com
;             502 Rue Orleanias
;             Deer Park, IL 60010                                   
;                                                                      
;***********************************************************************


	list		p=16f688		; list directive to define processor
	#include	<P16F688.inc>		; processor specific variable definitions
	
	__CONFIG    _CP_OFF & _CPD_OFF & _BOD_OFF & _PWRTE_ON & _WDT_OFF & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _FCMEN_OFF & _IESO_OFF


; '__CONFIG' directive is used to embed configuration data within .asm file.
; The labels following the directive are located in the respective .inc file.
; See respective data sheet for additional information on configuration word.






;***** VARIABLE DEFINITIONS
            CBLOCK      0x71    ; all banks vars
               w_temp		; variable used for context saving
               status_temp	; variable used for context saving
               pclath_temp	; variable used for context saving
               regA_dir         ; temp TRISA storage
               regC_dir         ; temp TRISC storage
               pindex           ; pattern inde
               input
            ENDC
               
            CBLOCK      0x21
				repeat
				IR_40k
				tx_on
				tx_1
				out0
				out1
				out2
				out3
				out4
				scount
				instate
				pindex2
				IR_in
				command
				address
				toggle
				IR_lhalf
				tbl2_hi
				tbl2_low
				button
				regA_out
                regC_out
				in0
				in1
				in2
				in3
				in4
				in5
				in6
				in7
				in8
                in9
				dcount
				b2l_out_hi
				b2l_out_low
				b2l_lcount
				b2l_in_red
				b2l_in_green
				count
				speed
				pspeed
				max_instate
				count1
                count2
                count3
				pattern
				tbl_hi
				tbl_low
				temp
				tbltype
                qcount
                IR_in_good
                dirty
                ploop
                out_address
                out_command
                out_toggle
                out_pattern
                demo_mode
                pclath_save
                temp2
                nextf
                nexti
                char_in
                tentry
                offset_low
                offset_hi
                eop
                char_color
                t4_count
                dchar
                lower
				random
            ENDC

NUM_TAB     EQU d'16'     ; number of tables

#undefine   mono
; #define     mono


LedOut macro word, bit, reg_dir, port
    btfsc       word, bit
    bcf         reg_dir, port
    endm

state_tbl macro num, zero, one, flag
Lstate#v(num)
    if (flag == 0)
       clrf button
    endif 
    if (flag == 1)
       bsf  button,0
    endif
 
    btfsc        IR_in,0
    goto         Lstate#v(num)a

    movlw        zero
    movwf        instate

    goto         Lcontinue
      
Lstate#v(num)a
    movlw        one
    movwf        instate

    goto         Lcontinue    
    endm

state2_tbl macro num, name, offset
Lstate#v(num)
;   save the 1st half of word
    movf         IR_in,w    ; save 1st half
    movwf        IR_lhalf
    incf         instate,f

    goto         Lcontinue
      
Lstate#v(num+1)
;   save the 2nd half of word
;   check for protocal error should get 01 or 10
    movf         IR_lhalf,w
    xorwf        IR_in,w
    btfss        STATUS, Z
    goto         Lstate#v(num+1)a
    clrf         instate    ; error state = zero
 
    goto         Lcontinue

;   set the correct bit in output vars.
Lstate#v(num+1)a
    btfsc        IR_in,0
    bcf          name, offset
    btfss        IR_in,0
    bsf          name, offset

    incf         instate,f    
    goto         Lcontinue
    endm


entry macro a0, a1, a2, a3, a4, a5, a6, a7, a8, a9
    retlw a0
    retlw a1
    retlw a2
    retlw a3
    retlw a4
    retlw a5
    retlw a6
    retlw a7
    retlw a8
    retlw a9
    endm

entry2 macro a0, a1
    retlw a0
    retlw a1
    endm

entry3 macro a0
    retlw a0
    endm

entrychar macro a0, a1, a2, a3, a4,
    retlw a0
    retlw a1
    retlw a2
    retlw a3
    retlw a4
    endm


tbltype1 macro speed
    retlw 0x01
    retlw speed
    endm

tbltype2 macro speed
    retlw 0x02
    retlw speed
    endm

tbltype3 macro speed
    retlw 0x03
    retlw speed
    endm

tbltype4 macro speed
    retlw 0x04
    retlw speed
    endm

tbltype5 macro speed
    retlw 0x05
    retlw speed
    endm

make_mtab macro 
    local i = 0
    while i <= NUM_TAB
      retlw HIGH table#v(i)
      retlw LOW table#v(i) 
i += 1
    endw
    endm

make_state_jump_table macro max_state
    local i = 0
    while i <= max_state
      retlw HIGH Lstate#v(i)
      retlw LOW Lstate#v(i) 
i += 1
    endw
    endm

   
;**********************************************************************
	ORG		0x000			; processor reset vector
  	goto		main			; go to beginning of program

	ORG		0x004			; interrupt vector location
	movwf		w_temp			; save off current W register contents
	movf		STATUS,w		; move status register into W register
	movwf		status_temp		; save off contents of STATUS register
	movf		PCLATH,w		; move pclath register into W register
	movwf		pclath_temp		; save off contents of PCLATH register

    ;   Bank 0
    bcf         STATUS, RP0
    bcf         STATUS, RP1

;
;   send/next pattern counter
;
    nop
    incfsz      count1,f
    goto        skipf

    incfsz      count2,f
    goto        skipf

    bsf         nexti,0

    incfsz      count3,f
    goto        skipf

    bsf         nextf,0

skipf
;   set tick interupt to 889ms for IR RC5 protocal 1/2 digit.
;   an Interupt every 889us /w 8mhz internal os

    movlw       D'36'      ; 38k            ; E20
    movwf       TMR0                        ; E21

    movf        scount,w                    ; F22
    addlw       0                           ; F23
    btfsc       STATUS,Z                    ; F24
    goto        send_nothing                ; F25

    btfss       tx_1,0                      ; F26
    goto        send_zero                   ; F27

;--------------------------------------------------------------------------------------
send_one  ; we are slaming the 36K IR siginal, its like a busy wait!
;--------------------------------------------------------------------------------------
; Portability : hard coded IR output line
;               hard coded timing loops
;

    bsf         PORTA,4                     ; A01

    call        get_next_tx_bit             ; 2 + 11 cycles  A04-A16
;    mult_nop d'9' ; A17-A25
    movlw       01h
    call        qdelay           ; 6 + (1 * 3) = 9 ; A17-A25
    nop

    movlw           0x1E                    ; C26  (23)
    movwf           repeat                  ; C27
repeat_loop
    bcf         PORTA,4                         ; D01

;    mult_nop d'26' ; D02-D27
    movlw       07h
    call        qdelay           ; 6 + (7 * 3) = 27 ; D02-D27

    bsf         PORTA,4                         ; E01  (27)

;    mult_nop d'23' ; E02-E24
    movlw       06h              ; 6+(6 * 3) = 22 ; E02-E24
    call qdelay

    decfsz      repeat,f                    ; E25      (24)
    goto        repeat_loop                    ; E26/E27

    nop                                     ;E27
    goto        exit_tx

;--------------------------------------------------------------------------------------
send_zero  ; we are going to release all the CPU time to the main line for display.
;--------------------------------------------------------------------------------------
; Portability : clean


    call        get_next_tx_bit             ; 2 + 11 cycles  A04-A16

    goto        exit_tx                     ; E25/26

;--------------------------------------------------------------------------------------
send_nothing ; need to eat 1/2 of the CPU time to make main line smooth.
;--------------------------------------------------------------------------------------
; Portability : timing loop

    call        read_ir_in
    call        read_button_in
    call        get_state_tbl_hi_address
    call        get_state_tbl_low_address
    call        go_state_table
    call        set_max_state

    movlw           0x0e                    ; A26  (removed one cycle, state machine has some overhead.
    movwf           repeat                  ; A27
repeat_loop3

    movlw       d'15'              ; 6+(15 * 3) = 51 ; B01-B27 + C01-C24
    call qdelay

    decfsz      repeat,f                    ; C25
    goto        repeat_loop3                    ; C26/C27

    nop                                     ; C27
 
;--------------------------------------------------------------------------------------
exit_tx
;--------------------------------------------------------------------------------------
; Portability : PIC chip specific
;

;   clear timer0 interupt
    bcf         INTCON,T0IF                 ; E26
    bcf         PORTA,4                     ; F01

; restore state, return form interupt

exit_int
	movf		pclath_temp,w		; retrieve copy of PCLATH register
	movwf		PCLATH			; restore pre-isr PCLATH register contents	
	movf		status_temp,w		; retrieve copy of STATUS register
	movwf		STATUS			; restore pre-isr STATUS register contents
	swapf		w_temp,f
	swapf		w_temp,w		; restore pre-isr W register contents
	retfie					; return from interrupt


;--------------------------------------------------------------------------------------
read_button_in   ; 5 cycles
;--------------------------------------------------------------------------------------
; Portability : hardcoded button port
;

    bcf         button,0         ; 1
    btfss       PORTA,5          ; 2        
    bsf         button,0         ; 3       

    return                        

;--------------------------------------------------------------------------------------
read_ir_in   ; 5 cycles
;--------------------------------------------------------------------------------------
; Portability : hardcoded IR input port
;

    bcf         IR_in,0         ; 1
    btfsc       PORTA,3         ; 2
    bsf         IR_in,0         ; 3

;    movf        tx_1,w   ; loopback for testing HACK!!        ;1
;    xorlw       0x01       ; loopback for testing HACK!!      ;2
;    movwf       IR_in      ; loopback for testing HACK!!      ;3
    return                                                     ; 4/5

;--------------------------------------------------------------------------------------
set_max_state   ; 8 cycles
;--------------------------------------------------------------------------------------
; Portability : clean
;

    movf         max_instate,w   ; 1
    subwf        instate,w       ; 2
    btfss        STATUS,C        ; 3
    goto         smi_l1          ; 4   4/5
    movf         instate,w       ; 5
    movwf        max_instate     ; 6
    return                       ; 7/8
smi_l1
    nop                          ; 6
    return                       ; 7/8

;--------------------------------------------------------------------------------------
get_next_tx_bit   ; 11 cycles
;--------------------------------------------------------------------------------------
; Portability : clean
;

    bcf         tx_1,0                                 ;1
    decf        scount,f                               ;2
    bcf         STATUS, C   ; clear carry bit          ;3
    rrf         out3,f                                 ;4
    rrf         out2,f                                 ;5
    rrf         out1,f                                 ;6
    rrf         out0,f                                 ;7
    btfsc       STATUS, C                              ;8
    bsf         tx_1,0                                 ;9
 
;    movf        tx_1,w   ; loopback for testing HACK!!
;    xorlw       0x01       ; loopback for testing HACK!!
;    movwf       tx_1      ; loopback for testing HACK!!
;    mult_nop d'6'         ; for debuging
  
    return                                             ;10/11

;**********************************************************************
qdelay
;**********************************************************************
; Portability : clean
;

    movwf       qcount
delay_loop2
    decfsz      qcount,1
    goto        delay_loop2
    return

;--------------------------------------------------------------------------------------
get_state_tbl_hi_address ; 17 cycles
;--------------------------------------------------------------------------------------
; Portability : clean
;
    bcf          STATUS,0      ; clear carry flag             ; 1
    rlf          instate,w       ; entry = table * 2          ; 2
    movwf        pindex2                                      ; 3
    call         table_lookup2                                ; 14 (9+2)
    movwf        tbl2_hi                                      ; 15
    return                                                    ; 16/17

;--------------------------------------------------------------------------------------
get_state_tbl_low_address  ; 18 cycles
;--------------------------------------------------------------------------------------
; Portability : clean
;
    bcf          STATUS,0      ; clear carry flag (just in case)  ; 1
    rlf          instate,w     ; entry = table * 2 + 1            ; 2
    addlw        1                                                ; 3
    movwf        pindex2                                          ; 4
    call         table_lookup2                                    ; 15 (9+2)
    movwf        tbl2_low                                         ; 16
    return                                                        ; 17/18

;**********************************************************************
table_lookup2  ; 9 cycles
;**********************************************************************
; Portability : clean
; 
   movlw        HIGH state_jump_table                 ; 1
    movwf        PCLATH                                ; 2
    movf         pindex2,w                             ; 3
    addlw        LOW state_jump_table                  ; 4
    btfsc        STATUS,0     ; check carry flag       ; 5
    incf         PCLATH,f                              ; 6
    movwf        PCL                                   ; 7 + 2

;**********************************************************************
go_state_table  ; 18 cycles
;**********************************************************************
; Portability : clean
;
    movf         tbl2_hi,w    ; 1
    movwf        PCLATH       ; 2
    movf         tbl2_low,w   ; 3
    movwf        PCL          ; 4

;**********************************************************************
state_jump_table
;**********************************************************************
; Portability : clean
;
    make_state_jump_table d'35'

    state_tbl 0, 7, 1, -1             ; don't reset the button state
    state_tbl 1, 0, 2, -1
    state_tbl 2, 0, 3, -1
    state_tbl 3, 4, 3, -1
    state_tbl 4, 0, 5, -1
    state_tbl 5, d'11', 0, -1

    state_tbl 6, 0, 0, -1  ; junk state

    state_tbl 7, 8, 0, -1
    state_tbl 8, 9, 0, -1
    state_tbl 9, d'10', 0, -1
    state_tbl d'10', d'10', 0, -1      ; don't set button for 19 led!

    state2_tbl d'11', toggle, 0
    state2_tbl d'13', address, 4    
    state2_tbl d'15', address, 3    
    state2_tbl d'17', address, 2    
    state2_tbl d'19', address, 1    
    state2_tbl d'21', address, 0    
    state2_tbl d'23', command, 5    
    state2_tbl d'25', command, 4    
    state2_tbl d'27', command, 3    
    state2_tbl d'29', command, 2    
    state2_tbl d'31', command, 1    
    state2_tbl d'33', command, 0    

Lstate35
    clrf        instate
    bsf         IR_in_good,0    

Lcontinue
    return ; 17/18

;**********************************************************************
bin2led
;**********************************************************************
; Portability : clean
;
    clrf        b2l_out_hi
    clrf        b2l_out_low
    movlw       0x08
    movwf        b2l_lcount
b2l_loop
    bcf         STATUS, C
    rrf         b2l_in_green,f
    rlf         b2l_out_low,f
    rlf         b2l_out_hi,f
    rrf         b2l_in_red,f
    rlf         b2l_out_low,f
    rlf         b2l_out_hi,f

    decfsz      b2l_lcount,f
    goto        b2l_loop
    return

;**********************************************************************
rf_command_out
;**********************************************************************
; Portability : clean
;
    clrf        out1
    clrf        out2
    clrf        out3

;   start flag
    movlw       B'10100000'
    movwf       out0


;   toggle bit
    comf        out_toggle,f                ; toggle the toggle flag
    btfss       out_toggle,0
    bsf         out1,0
    btfsc       out_toggle,0
    bsf         out1,1

;   address bits
    btfss       out_address,4
    bsf         out1,2
    btfsc       out_address,4
    bsf         out1,3

    btfss       out_address,3
    bsf         out1,4
    btfsc       out_address,3
    bsf         out1,5

    btfss       out_address,2
    bsf         out1,6
    btfsc       out_address,2
    bsf         out1,7

    btfss       out_address,1
    bsf         out2,0
    btfsc       out_address,1
    bsf         out2,1

    btfss       out_address,0
    bsf         out2,2
    btfsc       out_address,0
    bsf         out2,3

;   command bits
    btfss       out_command,5
    bsf         out2,4
    btfsc       out_command,5
    bsf         out2,5

    btfss       out_command,4
    bsf         out2,6
    btfsc       out_command,4
    bsf         out2,7

    btfss       out_command,3
    bsf         out3,0
    btfsc       out_command,3
    bsf         out3,1

    btfss       out_command,2
    bsf         out3,2
    btfsc       out_command,2
    bsf         out3,3

    btfss       out_command,1
    bsf         out3,4
    btfsc       out_command,1
    bsf         out3,5

    btfss       out_command,0
    bsf         out3,6
    btfsc       out_command,0
    bsf         out3,7

    movlw       0x21
    movwf       scount

    return
                 
;**********************************************************************
delay
;**********************************************************************
; Portability : clean
;
    movwf       count
delay_loop
    nop
    decfsz      count,1
    goto        delay_loop
    return

;**********************************************************************
init_pattern
;**********************************************************************
; Portability : clean
;
    clrf         count1
    clrf         count2
    movlw       0xfD           ; next pattern in 2 minutes?
    movwf       count3
    movlw       0x05
    movwf       char_color
    return

;**********************************************************************
next_pattern
;**********************************************************************
; Portability : clean
;
    incf         pattern,f
    movf         pattern,w
    sublw        NUM_TAB+1  ; max pattern?
    btfsc        STATUS,2   ; z bit check
    clrf         pattern    ; reset to zero
    movfw        pattern
    movwf        out_pattern
    call         init_pattern
    return

;**********************************************************************
past_pattern
;**********************************************************************
; Portability : clean
;
    decf         pattern,f
    movf         pattern,w
    addlw        1  ; max pattern?
    btfss        STATUS,2   ; z bit check
    goto         past_exit
    movlw        NUM_TAB
    movwf        pattern
    movwf        out_pattern
past_exit
    call         init_pattern
    return

;**********************************************************************
set_pattern
;**********************************************************************
; Portability : clean
;
    movfw        command
    movwf        pattern
    movwf        out_pattern
set_loop
    addlw        0xff-NUM_TAB
    btfss        STATUS, C
    goto         set_exit

    movlw        NUM_TAB
    subwf        pattern
    movfw        pattern
    goto         set_loop

set_exit
    call         init_pattern
    return


;**********************************************************************
strobe_IO
;**********************************************************************
;
;   write to hardware
;
;
; A0 A1 A2 C0 C1 C2 C3 C4 C5 - Led Matrix IO

    bcf         INTCON, GIE      ; disable INTs

    bsf         STATUS, RP0  ; switch to bank 1
    bcf         STATUS, RP1

    movlw       07h          ; set led outputs to input
    iorwf       TRISA,f
    movlw       3fh
    iorwf       TRISC,f

    bcf         STATUS, RP0 ; switch to bank 0
    bcf         STATUS, RP1

    movlw       0f8h         ; clear output the last pattern.
    andwf       PORTA,f
    movlw       0c0h
    andwf       PORTC,f

    bsf         STATUS, RP0  ; switch to bank 1
    bcf         STATUS, RP1

    movf        regA_dir,w   ; set portA direction
    andwf       TRISA
    movf        regC_dir,w   ; set portC direction
    andwf       TRISC

    bcf         STATUS, RP0 ; switch to bank 0
    bcf         STATUS, RP1

    movf        regA_out,w   ; set portA output
    iorwf       PORTA
    movf        regC_out,w   ; set portC output
    iorwf       PORTC

    bsf          INTCON, GIE      ; enable INTs

    return

;**********************************************************************
default_IO
;**********************************************************************
;
;   set io vars to default state.
;
;
; A0 A1 A2 C0 C1 C2 C3 C4 C5 - Led Matrix IO

    movlw       07h
    iorwf       regA_dir, f
    movlw       3Fh
    iorwf       regC_dir, f

    movlw       0f8h
    andwf       regA_out, f
    movlw       0c0h
    andwf       regC_out, f

    return




;**********************************************************************
display
;**********************************************************************
; A0 A1 A2 C0 C1 C2 C3 C4 C5 - Led Matrix IO

#ifdef mono
; --------------------
;  Mono Hack
; --------------------
    movfw       in1
    iorwf       in0,f
    movfw       in3
    iorwf       in2,f
    movfw       in5;
    iorwf       in4,f
    movfw       in7
    iorwf       in6,f
    movfw       in9
    iorwf       in8,f

    clrf        in1
    clrf        in3
    clrf        in5
    clrf        in7
    movlw       0x80
    andwf       in9,f   ; can't trash the end of pattern flag!  
#endif mono

    movf        speed,w
    movwf       dcount

; -----------------------
;  Strobe A0
; -----------------------

dloop
    call default_IO
;
;   A1 Led 12R
    LedOut in3, 3, regA_dir, 1

;   A2 Led 11R
    LedOut in5, 3, regA_dir, 2

;   C0 Led 10R
    LedOut in7, 3, regC_dir, 0

;   C1 Led 9R
    LedOut in9, 3, regC_dir, 1

;   C2 Led 18R
    LedOut in0, 3, regC_dir, 2

;   C3 Led 18R
    LedOut in0, 5, regC_dir, 3

;   C4 Led 17R
    LedOut in0, 6, regC_dir, 4

;   C5 Led 17R
    LedOut in0, 4, regC_dir, 5

;   strobe A0
    bcf         regA_dir, 0
    bsf         regA_out, 0

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strobe A1
; -----------------------
    call default_IO
;
;   A0 Led 12G
    LedOut in2, 3, regA_dir, 0

;   A2 Led 1R
    LedOut in2, 0, regA_dir, 2

;   C0 Led 2R
    LedOut in2, 1, regC_dir, 0

;   C1 Led 14R
    LedOut in2, 2, regC_dir, 1

;   C2 No led
    bsf            regC_dir, 2

;   C3 Led 13R
    LedOut in2, 5, regC_dir, 3

;   C4 Led 19R
    LedOut in2, 6, regC_dir, 4

;   C5 Led 19R
    LedOut in2, 4, regC_dir, 5

;   strobe A1
    bcf         regA_dir, 1
    bsf         regA_out, 1

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob A2
; -----------------------

    call default_IO
;
;   A0 Led 11G
    LedOut in4, 3, regA_dir, 0

;   A1 Led 1G
    LedOut in3, 0, regA_dir, 1

;   C0 Led 3R
    LedOut in4, 1, regC_dir, 0

;   C1 Led 4R
    LedOut in4, 2, regC_dir, 1

;   C2 Led 15R
    LedOut in0, 0, regC_dir, 2

;   C3 Led 15R
    LedOut in4, 5, regC_dir, 3

;   C4 Led 16R
    LedOut in4, 6, regC_dir, 4

;   C5 Led 16R
    LedOut in4, 4, regC_dir, 5

;   strob A2
    bcf         regA_dir, 2
    bsf         regA_out, 2

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C0
; -----------------------

    call default_IO
;
;   A0 Led 10G
    LedOut in6, 3, regA_dir, 0

;   A1 Led 2G
    LedOut in3, 1, regA_dir, 1

;   A2 Led 3G
    LedOut in5, 1, regA_dir, 2

;   C1 Led 5G
    LedOut in6, 2, regC_dir, 1

;   C2 Led 6R
    LedOut in0, 1, regC_dir, 2

;   C3 Led 6R
    LedOut in6, 5, regC_dir, 3

;   C4 Led 7R
    LedOut in6, 6, regC_dir, 4

;   C5 Led 7R
    LedOut in6, 4, regC_dir, 5

;   strob C0
    bcf         regC_dir, 0
    bsf         regC_out, 0

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C1
; -----------------------

    call default_IO
;
;   A0 Led 9G
    LedOut in8, 3, regA_dir, 0

;   A1 Led 14G
    LedOut in3, 2, regA_dir, 1

;   A2 Led 4G
    LedOut in5, 2, regA_dir, 2

;   C0 Led 5G
    LedOut in7, 2, regC_dir, 0

;   C2 no Led
    LedOut in0, 2, regC_dir, 2

;   C3 no Led
    LedOut in8, 5, regC_dir, 3

;   C4 Led 8R
    LedOut in8, 6, regC_dir, 4

;   C5 Led 8R
    LedOut in8, 4, regC_dir, 5

;   strob C1
    bcf         regC_dir, 1
    bsf         regC_out, 1

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C2
; -----------------------

    call default_IO
;
;   A0 18G
    LedOut in1, 3, regA_dir, 0

;   A1 No led
    bsf            regA_dir, 1

;   A2 15G
    LedOut in1, 0, regA_dir, 2

;   C0 6G
    LedOut in1, 1, regC_dir, 0

;   C1 No Led
    LedOut in1, 2, regC_dir, 1

;   C3 No Led
    LedOut in7, 0, regC_dir, 3

;   C4 No Led
    LedOut in5, 0, regC_dir, 4

;   C5 No Led
    LedOut in9, 0, regC_dir, 5

;   strob C2
    bcf         regC_dir, 2
    bsf         regC_out, 2

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C3
; -----------------------

    call default_IO
;
;   A0 18G
    LedOut in1, 5, regA_dir, 0

;   A1 13G
    LedOut in3, 5, regA_dir, 1

;   A2 15G
    LedOut in5, 5, regA_dir, 2

;   C0 6G
    LedOut in7, 5, regC_dir, 0

;   C1 No Led
    LedOut in9, 5, regC_dir, 1

;   C2 No Led
    LedOut in6, 0, regC_dir, 2

;   C4 No Led
    LedOut in6, 1, regC_dir, 4

;   C5 No Led
    LedOut in9, 2, regC_dir, 5

;   strob C3
    bcf         regC_dir, 3
    bsf         regC_out, 3

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C4
; -----------------------

    call default_IO
;
;   A0 Led 17G
    LedOut in1, 6, regA_dir, 0

;   A1 Led 19G
    LedOut in3, 6, regA_dir, 1

;   A2 Led 16G
    LedOut in5, 6, regA_dir, 2

;   C0 Led 7G
    LedOut in7, 6, regC_dir, 0

;   C1 Led 8G
    LedOut in9, 6, regC_dir, 1

;   C2 No Led
    LedOut in4, 0, regC_dir, 2

;   C3 No Led
    LedOut in7, 1, regC_dir, 3

;   C5 No Led
    LedOut in9, 1, regC_dir, 5

;   strob C4
    bcf         regC_dir, 4
    bsf         regC_out, 4

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay

; -----------------------
;  Strob C5
; -----------------------

    call default_IO
;
;   A0 Led 17G
    LedOut in1, 4, regA_dir, 0

;   A1 Led 19G
    LedOut in3, 4, regA_dir, 1

;   A2 Led 16G
    LedOut in5, 4, regA_dir, 2

;   C0 Led 7G
    LedOut in7, 4, regC_dir, 0

;   C1 Led 8G
    LedOut in9, 4, regC_dir, 1

;   C2 No Led
    LedOut in8, 0, regC_dir, 2

;   C3 No Led
    LedOut in8, 2, regC_dir, 3

;   C4 No Led
    LedOut in8, 1, regC_dir, 4

;   strob C5
    bcf         regC_dir, 5
    bsf         regC_out, 5

;   write to hardware
    call        strobe_IO

    movlw       0B0h
    call        delay



    decfsz      dcount,1
    goto        dloop
    
    return
;**********************************************************************
display_pattern_id
;**********************************************************************
; Portability : coded per board IO
;
    movf         pattern,w
    addlw        1
    movwf        in0
    clrf         in1
    clrf         in2          
    clrf         in3          
    clrf         in4          
    clrf         in5          
    clrf         in6          
    clrf         in7          
    clrf         in8          
    clrf         in9          

    movlw        0xff
    btfsc        demo_mode,0
    movwf        in8    

    movlw        0x7F
    movwf        speed
    call         display
    return


;**********************************************************************
setup_pattern_table
;**********************************************************************
; Portability : clean
;

stop2
    movf         pattern,w
    btfss        STATUS,Z
    goto         slabel1
;
;   if patten is zero, check eeprom and verify 1st entry is not a 0x00
;   if it is switch to next pattern.

    clrw
    call         read_eeprom
    movwf        temp

    clrf         demo_mode
    btfsc        temp,7
    bsf          demo_mode,0

    movfw        temp
    btfss        STATUS,Z
    goto         sreturn
    
    call         next_pattern
    goto         stop2

slabel1
;
;   Set the pattern (table)
;
    movlw        HIGH MasterTable
    movwf        tbl_hi
    movlw        LOW MasterTable
    movwf        tbl_low

;   get the high address for the pattern table
    bcf          STATUS,0      ; clear carry flag
    rlf          pattern,w     ; entry = table * 2
    movwf        pindex
    bcf          INTCON, GIE      ; disable INTs
    call         table_lookup
    bsf          INTCON, GIE      ; enable INTs
    movwf        temp          ; can't put it in tbl_high yet

;   get the low address for the pattern table
    bcf          STATUS,0      ; clear carry flag (just in case)
    rlf          pattern,w     ; entry = table * 2 + 1
    addlw        1
    movwf        pindex
    bcf          INTCON, GIE      ; disable INTs
    call         table_lookup
    bsf          INTCON, GIE      ; enable INTs
    movwf        tbl_low

    movf         temp,w
    movwf        tbl_hi

sreturn
    return

;**********************************************************************
get_next_pattern_entry
;**********************************************************************
; Portability : clean
;
    movf         pattern,w
    btfss        STATUS,Z
    goto         glabel1

    incf         pindex,w          ; pindex + 1 (1st location in eeprom is special.
    call         read_eeprom
    
    goto         greturn
glabel1
    movfw        PCLATH
    movwf        pclath_save
    movf         pindex,w
    bcf          INTCON, GIE      ; disable INTs
    call         table_lookup
    movwf        temp2
    movfw        pclath_save
    movwf        PCLATH
    bsf          INTCON, GIE      ; enable INTs
    movfw        temp2

greturn
    incf         pindex,f
    return

;**********************************************************************
write_eeprom
;**********************************************************************
; Portability : clean
;
    bsf          STATUS, RP0      ; bank 1
    bcf          STATUS, RP1

    bsf          EECON1,WREN      ; allow writes to eeprom

    bcf          INTCON, GIE      ; disable INTs

    movf         pindex,w
    movwf        EEADR            ; load address

    movf         input,w
    movwf        EEDAT            ; lead data

    movlw        0x55             ; unlock
    movwf        EECON2
    movlw        0xAA
    movwf        EECON2

    bsf          EECON1,WR
    bsf          INTCON, GIE

    bcf          STATUS, RP0      ; bank 0
    bcf          STATUS, RP1

;   need a delay for the write of eeprom before we read it again, a call to the
;   display does the trick. A read before the write is done is bad!
;
    movlw        0xfF
    movwf        in0
    movwf        in1

    movlw        0x3f
    movwf        speed
    call         display   

    return

;**********************************************************************
read_eeprom
;**********************************************************************
; Portability : clean
;

;   address of eeprom to read passed in x reg
;   value read passed out in x reg
;

;    movf         pindex,w
   ;   switch bank 1
    bsf          STATUS, RP0  
    bcf          STATUS, RP1

    movwf        EEADR
    bcf          EECON1, 7  ; EEPDG
    bsf          EECON1, RD
    movf         EEDAT, w

;   switch bank 0
    bcf          STATUS, RP0
    bcf          STATUS, RP1

;    movwf        input
    return

;**********************************************************************
program_pattern  
;**********************************************************************
; Portability : coded per board IO
;
    clrf         pindex
    clrf         dirty
    clrf         ploop
    clrf         lower

    movlw       0x05
    movwf       pindex

;  flash all LEDs on 
    movlw       0xFF
    movwf       in0
    movwf       in1
    movwf       in2
    movwf       in3
    movwf       in4
    movwf       in5
    movwf       in6
    movwf       in7
    movwf       in8
    movwf       in9

    movlw       0x7f
    movwf       speed
    call        display   


    movf         pindex,w
    call         read_eeprom
    movwf        input

program_top
;   load current value and display    
    movfw        input
    call         display_char 

    movlw        0x1f                 ; this 1st display acts like a debounce for RF commands.
    movwf        speed
    call         display
    clrf         IR_in_good

program_loop

;   load current value and display    
    movfw        input
    call         display_char

    movlw        0x02
    movwf        speed
    call         display

    btfss        IR_in_good,0      ; have an IR command to process?
    goto         program_loop

    movf         address,f         ; remote giving "0" for address (TV)?
    btfss        STATUS,Z
    goto         program_loop

;    movlw        0x0D              ; remote menu putton (exit program mode)
;    subwf        command,w
;    btfss        STATUS,Z
;    goto         button0

;    btfsc        dirty,0
;    call         write_eeprom
;    clrf         input
;    clrf         dirty    
;    goto         program_exit

button0
    movlw        0x10              ; remote "volume +" button (next entry)
    subwf        command,w
    btfss        STATUS,Z
    goto         button1

    btfsc        dirty,0
    call         write_eeprom
    clrf         dirty
;
;   if rolled to 0xff, make 0xfe
    incf         pindex,f

;   if less then 0x05, make 0x05
    movfw        pindex
    addlw        0x01
    btfss        STATUS,Z
    goto         button0a

    movlw        0xfe
    movwf        pindex
    
button0a

;   check for max?

    movf         pindex,w
    call         read_eeprom
    movwf        input
    goto         program_top

button1
    movlw        0x11              ; remote "volume -" button (past entry)
    subwf        command,w
    btfss        STATUS,Z
    goto         button2

    btfsc        dirty,0
    call         write_eeprom
    clrf         dirty
    decf         pindex,f
;
;   if less then 0x05, make 0x05
    movfw        pindex
    sublw        0x05
    btfss        STATUS,C
    goto         button1a

    movlw        0x05
    movwf        pindex
    
button1a
    movf         pindex,w
    call         read_eeprom
    movwf        input
    goto         program_top

button2
    movlw        0x20              ; remote chan + putton (past character)
    subwf        command,w
    btfss        STATUS,Z
    goto         button3

    movlw        0x01            
    subwf        input,f

;   if less then 0x20
    movfw        input
    sublw        0x1f
    btfss        STATUS,C
    goto         button2a

    movlw        0x7a
    movwf        input

button2a    
    bsf          dirty,0
    goto         program_top


button3
    movlw        0x21              ; remote chan - putton (next character)
    subwf        command,w
    btfss        STATUS,Z
    goto         button4

    movlw        0x01              ; DDF
    addwf        input,f

;   if over 0x7a, next is 0x20
    movfw        input
    sublw        0x7a
    btfsc        STATUS,C
    goto         button3a

    movlw        0x20
    movwf        input

button3a
    
    bsf          dirty,0
    goto         program_top

button4
    movlw        0x01              ; remote "1" button color1
    subwf        command,w
    btfss        STATUS,Z
    goto         button5

    movlw        0x01
    movwf        input
    
    bsf          dirty,0
    goto         program_top

button5
    movlw        0x02              ; remote "2" button color2
    subwf        command,w
    btfss        STATUS,Z
    goto         button6

    movlw        0x02
    movwf        input
    
    bsf          dirty,0
    goto         program_top

button6
    movlw        0x03              ; remote "3" button color1 & color2
    subwf        command,w
    btfss        STATUS,Z
    goto         button7

    movlw        0x05
    movwf        input
    
    bsf          dirty,0
    goto         program_top


button7
    movlw        0x09              ; remote "9" button End-of-Pattern
    subwf        command,w
    btfss        STATUS,Z
    goto         button8

    movlw        0x00
    movwf        input
    
    bsf          dirty,0
    goto         program_top

button8
    movlw        0x00              ; remote "0" button (0 - Upper/Lower toggle)
    subwf        command,w
    btfss        STATUS,Z
    goto         button9

case_toggle
    movfw        input
    sublw        0x60
    btfsc        STATUS,C
    goto         make_lower

; make_lower
    movlw        0x20
    subwf        input,f
    bsf          lower,0
    goto         end_tcase

make_lower
    movlw        0x20
    addwf        input,f
    clrf         lower

end_tcase
    bsf          dirty,0
    goto         program_top

button9
    goto         program_loop

;**********************************************************************
toggle_demo
;**********************************************************************
    clrw
    call         read_eeprom
    movwf        temp

    btfss        temp,7          ;
    goto         tdemo_on

;   toggle off
    bcf          temp,7
    clrf         demo_mode
    goto         tdemo_cont

tdemo_on
    bsf          temp,7
    bsf          demo_mode,0

tdemo_cont
    clrf         pindex
    movfw        temp
    movwf        input
    call         write_eeprom

    call         display_pattern_id
    call         display_pattern_id
    
    return


;**********************************************************************
pattern_driver
;**********************************************************************
; Portability : coded per board IO
;
ptop


    call         setup_pattern_table

    clrf         pindex

;   load the table type  (Currently only 1 type)
    call         get_next_pattern_entry
    movwf        tbltype

;   load the speed (someday this will be a table type 1 only thing.)
    call         get_next_pattern_entry
    movwf        pspeed
    incf         pspeed,f          ; adjust so 0 is fastest, not treated like speed 256

ploop0

;   Check for IR command in

    btfss        IR_in_good,0
    goto         button_check

blinkie_check
    movlw        0x07              ; blinkines use address 0x07 (Experimental)
    subwf        address,w
    btfss        STATUS,Z
    goto         remote_check    

    call         set_pattern
    clrf         IR_in_good
    goto         ptop

remote_check
    movf         address,f         ; remote giving "0" for address (TV)?
    btfss        STATUS,Z
    goto         button_check

    movlw        0x20              ; remote curser right (pattern ++)
    subwf        command,w
    btfsc        STATUS,Z
    call         next_pattern

    movlw        0x21              ; remote curser left (pattern --)
    subwf        command,w
    btfsc        STATUS,Z
    call         past_pattern

    movlw        0x00              ; remote "0" key  (pattern = 0)
    subwf        command,w
    btfsc        STATUS,Z
    clrf         pattern

    movlw        0x0c              ; remote on/off key (togle demo mode!)
    subwf        command,w
    btfsc        STATUS,Z
    call         toggle_demo

    clrf         IR_in_good
    goto         ptop

button_check
ploop1
    btfss        button,0    ; if putton pressed show pattern id
    goto         ploop2

    call         display_pattern_id
    call         display_pattern_id

    btfss        button,0    ; if putton still pressed inc pattern
    goto         ploop2

ploop1a
    call         next_pattern
    call         display_pattern_id

    btfsc        button,0
    goto         ploop1a
    goto         ptop
    
ploop2

    movlw        0x01              ; pattern1
    subwf        tbltype,w
    btfsc        STATUS,Z
    call         table_type_1

    movlw        0x02              ; pattern2
    subwf        tbltype,w
    btfsc        STATUS,Z
    call         table_type_2

    movlw        0x03              ; pattern2
    subwf        tbltype,w
    btfsc        STATUS,Z
    call         table_type_3

    movlw        0x04              ; pattern2
    subwf        tbltype,w
    btfsc        STATUS,Z
    call         table_type_4

    movlw        0x05              ; pattern2
    subwf        tbltype,w
    btfsc        STATUS,Z
    call         table_type_5

    movf         pspeed,w
    movwf        speed
    call         display

;   count down till net IR signel
    btfsc       nexti,0
    goto        test_demo_mode   

;   make sure we are not sending a IR signel
    movf        scount,w                    ; F22
    addlw       0                           ; F23
    btfss       STATUS,Z                    ; F24
    goto        test_demo_mode

    movlw       0x07                        ; address 0x07 "Experimental"
    movwf       out_address
    movfw       out_pattern
    movwf       out_command                 ; copy out_pettern

    call        rf_command_out

test_demo_mode

;   count down till next demo mode change
    btfss       nextf,0
    goto        ploop3

;   demo mode on?
    clrf        nextf
    btfss       demo_mode,0
    goto        ploop3

;   next pattern
    movlw       0xfD           ; next pattern in 2 minutes?
    movwf       count3
    call        next_pattern
    goto        ptop
   
ploop3
;   check for end of pattern
    btfss        in9,7
    goto ploop0

    goto        ptop

;**********************************************************************
table_type_1
;**********************************************************************

    call         get_next_pattern_entry
    movwf        in0
  
    call         get_next_pattern_entry
    movwf        in1

    call         get_next_pattern_entry
    movwf        in2

    call         get_next_pattern_entry
    movwf        in3
 
    call         get_next_pattern_entry
    movwf        in4

    call         get_next_pattern_entry
    movwf        in5

    call         get_next_pattern_entry
    movwf        in6

    call         get_next_pattern_entry
    movwf        in7

    call         get_next_pattern_entry
    movwf        in8

    call         get_next_pattern_entry
    movwf        in9

    return

;**********************************************************************
table_type_2
;**********************************************************************

    call shift_left

    call  get_next_pattern_entry
    movwf in8

    call  get_next_pattern_entry
    movwf in9

    return

;**********************************************************************
table_type_3
;**********************************************************************

   call  get_next_pattern_entry

;
;  offset = chartabel + (char_in - 0x20) * 5 + char_offset
;  offset = chartabel + (char_in - 0x20) + ((char_in - 0x21) * 4) + char_offset
;  offset = chartabel + (char_in - 0x20) + ((char_in - 0x21) <<2) + char_offset
;  12 bit            12 bit      7 bit              9 bit                    4 bit


   movwf        char_in
   movf         char_in,f         ; test for zero

   btfsc        STATUS,Z
   goto         end_string

   movlw        0x20              ; color code?
   subwf        char_in,w
   
   btfsc        STATUS,C
   goto         skipt3

   movf         char_in,w         ; get color, then pull next entry
   movwf        char_color
   goto         table_type_3

skipt3
; char_in = char_in - 0x20
   movlw        0x20              
   subwf        char_in,f        

; offset = chartabel
   movlw        LOW chartable     
   movwf        offset_low 
   movlw        HIGH chartable
   movwf        offset_hi

; offset = offset + char_in
   movf         char_in,w           
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f   

; char_in << 2 ; if carry offset_hi++         
   bcf          STATUS, C         ; clear carry flag just in case
   rlf          char_in, f
   rlf          char_in, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f 

; offset = offset + char_in 
   movf         char_in,w           ; offset = offset + char_in
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f     

looper

   bcf          INTCON, GIE      ; disable INTs
   call         jump_address
   movwf        tentry

   movf         pclath_save,w
   movwf        PCLATH
   bsf          INTCON, GIE      ; enable INTs

; -------------------
    call shift_left

    movf  tentry, w
    movwf in8

    movf  tentry, w
    movwf in9

   btfss        char_color,0     ; color adjust
   clrf         in8

   btfss        char_color,2
   clrf         in9

   btfsc        char_color,1
   comf         in8,f

   btfsc        char_color,3
   comf         in9,f
   
; ------------------

; offset = offset + 1
   movlw        1
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f

   movf         pspeed,w
   movwf        speed
   call         display

   bcf          in9,7             ; clear end of patter flag
   btfss        tentry,7
   goto         looper

; -------------------
    call shift_left

    clrf in8

    clrf in9

   btfsc        char_color,1     ; color adjust
   comf         in8,f

   btfsc        char_color,3
   comf         in9,f
; ------------------

    goto end_char

end_string
   bsf          in9,7             ; set end of patter flag

end_char
   return

;**********************************************************************
display_char 
;**********************************************************************
   movwf        dchar             ; input in "w", save in dchar  

   movf         dchar,f         ; test for zero
   btfss        STATUS,Z
   goto         dchar_less

   movlw        0x7c              ; if end of pattern show char "#"
   movwf        dchar
   goto         dchar_printable

dchar_less
   movlw        0x20              ; is the char less them 0x20? color code?
   subwf        dchar,w
   btfsc        STATUS,C
   goto         dchar_tcont

   movf         dchar,w         ; get color, then switch to box char from printing (0x7b)
   movwf        char_color
   movlw        0x7b
   movwf        dchar
   goto         dchar_printable

dchar_tcont
   movlw        0x7b              ; is the char greater them 0x7b?
   subwf        dchar,w
   btfss        STATUS,C
   goto         dchar_printable

   movlw        0x7d
   movwf        dchar  

dchar_printable
; char_in = char_in - 0x20
   movlw        0x20              
   subwf        dchar        

; offset = chartabel
   movlw        LOW chartable     
   movwf        offset_low 
   movlw        HIGH chartable
   movwf        offset_hi

; offset = offset + char_in
   movf         dchar,w           
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f   

; char_in << 2 ; if carry offset_hi++         
   bcf          STATUS, C         ; clear carry flag just in case
   rlf          dchar, f
   rlf          dchar, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f 

; offset = offset + char_in 
   movf         dchar,w           ; offset = offset + char_in
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f     


    movlw       0x05
    movwf       t4_count

t4_looper

   bcf          INTCON, GIE      ; disable INTs
   call         jump_address
   movwf        tentry

   movf         pclath_save,w
   movwf        PCLATH
   bsf          INTCON, GIE      ; enable INTs

; -------------------
    call shift_left

    movf  tentry, w
    movwf in8

    movf  tentry, w
    movwf in9

   btfss        char_color,0     ; color adjust
   clrf         in8

   btfss        char_color,2
   clrf         in9

   btfsc        char_color,1
   comf         in8,f

   btfsc        char_color,3
   comf         in9,f
   
; ------------------

; offset = offset + 1
   movlw        1
   addwf        offset_low, f
   btfsc        STATUS,C          ; check carry flag
   incf         offset_hi,f

   bcf          in9,7             ; clear end of patter flag
   decfsz       t4_count,f
   goto         t4_looper

   return

;**********************************************************************
table_type_4
;**********************************************************************

   clrf         in0
   clrf         in1
   clrf         in2
   clrf         in3
   clrf         in4
   clrf         in5
   clrf         in6
   clrf         in7
   clrf         in8
   clrf         in9

   movlw        0x03
   movwf        speed
   call         display

   call  get_next_pattern_entry

;
;  offset = chartabel + (char_in - 0x20) * 5 + char_offset
;  offset = chartabel + (char_in - 0x20) + ((char_in - 0x21) * 4) + char_offset
;  offset = chartabel + (char_in - 0x20) + ((char_in - 0x21) <<2) + char_offset
;  12 bit            12 bit      7 bit              9 bit                    4 bit


   movwf        char_in

   movf         char_in,f         ; test for zero
   btfsc        STATUS,Z
   goto         t4_end_string

   movlw        0x20              ; color code?
   subwf        char_in,w
   btfsc        STATUS,C
   goto         t4_skipt3

   movf         char_in,w         ; get color, then pull next entry
   movwf        char_color
   goto         table_type_4

t4_skipt3
   movfw        char_in
   call         display_char
   goto end_char

t4_end_string
   bsf          in9,7             ; set end of patter flag

t4_end_char
   return

;**********************************************************************
gen_random
;**********************************************************************
   RLF     random,W
   RLF     random,W
   BTFSC   random,4
   XORLW   1
   BTFSC   random,5
   XORLW   1
   BTFSC   random,3
   XORLW   1
   MOVWF   random
   return


;**********************************************************************
table_type_5
;**********************************************************************

    call  gen_random
	movwf in3
    call  gen_random
	movwf in8
    call  gen_random
	movwf in5
    call  gen_random
	movwf in0
    call  gen_random
	movwf in4
    call  gen_random
	movwf in1
    call  gen_random
	movwf in7
    call  gen_random
	movwf in2
    call  gen_random
	movwf in9
    call  gen_random
	movwf in6

#ifdef mono
    clrf in1
    clrf in3
    clrf in5
    clrf in7
    clrf in9
#endif

    return

;**********************************************************************
shift_left
;**********************************************************************
; Portability : clean
;
    movf in2,w
    movwf in0

    movf in3,w
    movwf in1

    movf in4,w
    movwf in2

    movf in5,w
    movwf in3

    movf in6,w
    movwf in4

    movf in7,w
    movwf in5

    movf in8,w
    movwf in6

    movf in9,w
    movwf in7

    return

;**********************************************************************
jump_address
;**********************************************************************
; Portability : clean
;
    movf         PCLATH,w
    movwf        pclath_save
    movf         offset_hi,w
    movwf        PCLATH

    movf         offset_low, w
    movwf        PCL

;
; M A I N
;
;**********************************************************************
main
;**********************************************************************

    clrf        PORTA
    clrf        PORTC
    clrf        regA_dir
    clrf        regC_dir
    clrf        regA_out
    clrf        regC_out
    clrf        instate
    clrf        IR_in
    clrf        toggle
    clrf        address
    clrf        command
    clrf        IR_lhalf
    clrf        max_instate
    clrf        TMR0
    clrf        button

    clrf        count1
    clrf        count2
    movlw       0xfD           ; next pattern in 4 minutes?
    movwf       count3


    movlw       0x05
    movwf       char_color

    movlw       0x1D
    movwf       random


    movlw       0xff
    movwf       b2l_in_green
    clrf        b2l_in_red
    call        bin2led
    nop

;   setting portC to digital
    movlw       07h
    movwf       CMCON0

;   Bank 1
    bsf         STATUS, RP0
    bcf         STATUS, RP1

;   setup timer0 & pullups
;                 +---------- disable pullups on portA (1)
;                 | +-------- source Int clock (o)
;                 | | +------ prescaler on timer0 (0)
;                 | | |+++---  prescaler set to 8X (010)
;                 | | |||| 
    movlw       b'10000010'
    movwf       OPTION_REG
    
; switch to 8mH clock
    bsf         OSCCON, IRCF0
    bsf         OSCCON, IRCF1
    bsf         OSCCON, IRCF2

;   enable pullups on portA
    bcf         OPTION_REG,NOT_RAPU

;   input on A5
    bsf          regA_dir,5
    clrf         WPUA       ; disable all pullups
    bsf          WPUA,5     ; enable pullup A5

;   setting portC to digital
    clrf        ANSEL

;   turn off comparators
    movlw       0x07
    movwf       CMCON0

;   setup timer0
    bcf         OPTION_REG,T0CS

;   Bank 0
    bcf         STATUS, RP0
    bcf         STATUS, RP1

;   enable timer0 interupt
    bsf          INTCON,T0IE
    bsf          INTCON,GIE

;   start with pattern 0
    clrf         pattern

;   input on A3
    bsf          regA_dir,3

;   start with pattern 0
    clrf         pattern

;  flash all LEDs on     This kills some time, allowing us to sample the button and IR-Rx
    movlw       0xFF
    movwf       in0
    movwf       in1
    movwf       in2
    movwf       in3
    movwf       in4
    movwf       in5
    movwf       in6
    movwf       in7
    movwf       in8
    movwf       in9

    movlw       0x7f
    movwf       speed
    call        display    

;   if button pressed durring powerup run IR_Tester
;   else run pattern driver
;
    btfss       button,0
    goto        pattern_driver
    call        toggle_demo
    goto        program_pattern 



;**********************************************************************
table_lookup
;**********************************************************************
; Portability : clean
;
    movf         tbl_hi,w
    movwf        PCLATH
    movf         pindex,w
    addwf        tbl_low,w
    btfsc        STATUS,0     ; check carry flag
    incf         PCLATH,f
    movwf        PCL

;**********************************************************************
MasterTable
;**********************************************************************
    make_mtab

;**********************************************************************
table0  ; 
;**********************************************************************

;**********************************************************************
table1  ; 
;**********************************************************************

    tbltype4 0x30

    entry3 0x04
    entry3 "D"
    entry3 "u"
    entry3 "c"
    entry3 "K"
    entry3 "o"
    entry3 "n"
    entry3 " "
    entry3 0x01
    entry3 "1"
    entry3 "6"
    entry3 " " 
    entry3 " "
    entry3 0x00


;**********************************************************************
table2  ; 
;**********************************************************************

    tbltype3 0x20

    entry3 0x04
    entry3 "D"
    entry3 "u"
    entry3 "c"
    entry3 "K"
    entry3 "o"
    entry3 "n"
    entry3 " "
    entry3 0x05
    entry3 "1"
    entry3 "6"
    entry3 " "
    entry3 0x01 
    entry3 " "
    entry3 "T"
    entry3 "h"
    entry3 "e"
    entry3 " " 
    entry3 "D"
    entry3 "u"
    entry3 "c"
    entry3 "k"
    entry3 " " 
    entry3 "S"
    entry3 "i"
    entry3 "d"
    entry3 "e"
    entry3 " " 
    entry3 "o"
    entry3 "f"
    entry3 " "
    entry3 "t"
    entry3 "h" 
    entry3 "e"
    entry3 " "
    entry3 "F" 
    entry3 "o"
    entry3 "r"
    entry3 "c"
    entry3 "e"
    entry3 " "
    entry3 0x00

;**********************************************************************
table3  ; 
;**********************************************************************

    tbltype1 0x30
    entry 0x07 , 0x00 , 0x00 , 0x02 , 0x00 , 0x00 , 0x00 , 0x00 , 0x1C , 0x00
    entry 0x07 , 0x00 , 0x00 , 0x00 , 0x00 , 0x01 , 0x00 , 0x00 , 0x0E , 0x00
    entry 0x07 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x02 , 0x07 , 0x00
    entry 0x0E , 0x00 , 0x00 , 0x00 , 0x00 , 0x04 , 0x00 , 0x00 , 0x0E , 0x00
    entry 0x1C , 0x00 , 0x00 , 0x08 , 0x00 , 0x00 , 0x00 , 0x00 , 0x1C , 0x00
    entry 0x1C , 0x00 , 0x00 , 0x00 , 0x00 , 0x10 , 0x00 , 0x00 , 0x38 , 0x00
    entry 0x1C , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x20 , 0x70 , 0x00
    entry 0x38 , 0x00 , 0x00 , 0x00 , 0x00 , 0x40 , 0x00 , 0x00 , 0x70 , 0x00
    entry 0x70 , 0x00 , 0x00 , 0x20 , 0x00 , 0x00 , 0x00 , 0x00 , 0x70 , 0x00
    entry 0x38 , 0x00 , 0x00 , 0x00 , 0x00 , 0x10 , 0x00 , 0x00 , 0x70 , 0x00
    entry 0x1C , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x08 , 0x1C , 0x00
    entry 0x0E , 0x00 , 0x00 , 0x00 , 0x00 , 0x04 , 0x00 , 0x00 , 0x1C , 0x80

;**********************************************************************
table4  ; 
;**********************************************************************

    tbltype3 0x20

    entry3 0x04
    entry3 "w"
    entry3 "w"
    entry3 "w"
    entry3 "."
    entry3 "2"
    entry3 "D"
    entry3 "K"
    entry3 "i"
    entry3 "t"
    entry3 "s"
    entry3 "." 
    entry3 "c"
    entry3 "o"
    entry3 "m"
    entry3 0x00

;**********************************************************************
table5  ; 
;**********************************************************************

    tbltype1 0x07

    entry 0xF8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L1
    entry 0x07, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L2
    entry 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L3
    entry 0x00, 0x01, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L4
    entry 0x00, 0x00, 0x0F, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L5
    entry 0x00, 0x00, 0x00, 0x7C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L6
    entry 0x00, 0x00, 0x00, 0x03, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00   ; L7
    entry 0x84, 0x21, 0x08, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L8
    entry 0x42, 0x10, 0x84, 0x21, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00   ; L9
    entry 0x21, 0x08, 0x42, 0x10, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00   ; L10
    entry 0x10, 0x84, 0x21, 0x08, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00   ; L11
    entry 0x08, 0x42, 0x10, 0x84, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00   ; L12
    entry 0x00, 0x00, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00   ; L13
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 0x00, 0x00, 0x00   ; L14
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xC0, 0x00, 0x00, 0x00   ; L15
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00   ; L16
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xF0, 0x00, 0x00   ; L17
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0F, 0x80, 0x00   ; L18
    entry 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7C, 0x00   ; L19
    entry 0x00, 0x00, 0x00, 0x00, 0x10, 0x84, 0x21, 0x08, 0x40, 0x00   ; L20
    entry 0x00, 0x00, 0x00, 0x00, 0x08, 0x42, 0x10, 0x84, 0x20, 0x00   ; L21
    entry 0x00, 0x00, 0x00, 0x00, 0x04, 0x21, 0x08, 0x42, 0x10, 0x00   ; L22
    entry 0x00, 0x00, 0x00, 0x00, 0x02, 0x10, 0x84, 0x21, 0x08, 0x00   ; L23
    entry 0x00, 0x00, 0x00, 0x00, 0x01, 0x08, 0x42, 0x10, 0x85, 0x80   ; L24

;**********************************************************************
table6  ; 
;**********************************************************************

    tbltype1 0x10
   entry	0x01	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x02	,	0x00	,	0x01	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x04	,	0x00	,	0x02	,	0x00	,	0x01	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x08	,	0x00	,	0x04	,	0x00	,	0x02	,	0x00	,	0x01	,	0x00	,	0x00	,	0x00
   entry	0x10	,	0x00	,	0x08	,	0x00	,	0x04	,	0x00	,	0x02	,	0x00	,	0x01	,	0x00
   entry	0x20	,	0x00	,	0x10	,	0x00	,	0x08	,	0x00	,	0x04	,	0x00	,	0x02	,	0x00
   entry	0x40	,	0x00	,	0x20	,	0x00	,	0x10	,	0x00	,	0x08	,	0x00	,	0x04	,	0x00
   entry	0x00	,	0x00	,	0x40	,	0x00	,	0x20	,	0x00	,	0x10	,	0x00	,	0x08	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x40	,	0x00	,	0x20	,	0x00	,	0x10	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x40	,	0x00	,	0x20	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x40	,	0x80

;**********************************************************************
table7  ; 
;**********************************************************************
   tbltype1 0x10
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x08	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x1C	,	0x00	,	0x14	,	0x00	,	0x1C	,	0x00	,	0x00	,	0x00
   entry	0x3E	,	0x00	,	0x22	,	0x00	,	0x22	,	0x00	,	0x22	,	0x00	,	0x3E	,	0x00
   entry	0x41	,	0x00	,	0x41	,	0x00	,	0x41	,	0x00	,	0x41	,	0x00	,	0x41	,	0x80

;**********************************************************************
table8  ; 
;**********************************************************************

    tbltype3 0x20

    entry3 0x05
    entry3 "S"
    entry3 "o"
    entry3 " "
    entry3 "m"
    entry3 "a"
    entry3 "n"
    entry3 "y"
    entry3 " "
    entry3 "B"
    entry3 "l"
    entry3 "i" 
    entry3 "n"
    entry3 "k"
    entry3 "i"
    entry3 "e"
    entry3 "s"
    entry3 ","
    entry3 " "
    entry3 "s"
    entry3 "o"
    entry3 " "
    entry3 "l"
    entry3 "i"
    entry3 "t" 
    entry3 "t"
    entry3 "l"
    entry3 "e"
    entry3 " "
    entry3 "t"
    entry3 "i"
    entry3 "m" 
    entry3 "e"
    entry3 " "
    entry3 0x00

;**********************************************************************
table9  ; 
;**********************************************************************
   tbltype1 0x10
   entry	0x04	,	0x20	,	0x02	,	0x10	,	0x41	,	0x08	,	0x02	,	0x10	,	0x04	,	0x20
   entry	0x02	,	0x10	,	0x41	,	0x08	,	0x20	,	0x04	,	0x41	,	0x08	,	0x02	,	0x10
   entry	0x41	,	0x08	,	0x20	,	0x04	,	0x10	,	0x02	,	0x20	,	0x04	,	0x41	,	0x08
   entry	0x20	,	0x04	,	0x10	,	0x02	,	0x08	,	0x41	,	0x10	,	0x02	,	0x20	,	0x04
   entry	0x10	,	0x02	,	0x08	,	0x41	,	0x04	,	0x20	,	0x08	,	0x41	,	0x10	,	0x02
   entry	0x08	,	0x41	,	0x04	,	0x20	,	0x02	,	0x10	,	0x04	,	0x20	,	0x08	,	0xC1

;**********************************************************************
table10  ; 
;**********************************************************************
   tbltype1 0x20

   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x63	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x10	,	0x00	,	0x10	,	0x00	,	0x00	,	0x00	,	0x04	,	0x00	,	0x04	,	0x00
   entry	0x04	,	0x00	,	0x04	,	0x00	,	0x00	,	0x00	,	0x10	,	0x00	,	0x10	,	0x80

;**********************************************************************
table11  ; 
;**********************************************************************

    tbltype4 0x30

    entry3 0x04
    entry3 "w"
    entry3 "w"
    entry3 "w"
    entry3 "."
    entry3 "2"
    entry3 "D"
    entry3 "K"
    entry3 "i"
    entry3 "t"
    entry3 "s"
    entry3 "." 
    entry3 "c"
    entry3 "o"
    entry3 "m"
    entry3 " "
    entry3 " "
    entry3 0x00

;**********************************************************************
table12  ; 
;**********************************************************************
   tbltype1 0x10

   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x40	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x20	,	0x00	,	0x60	,	0x00	,	0x20	,	0x00	,	0x00
   entry	0x00	,	0x10	,	0x00	,	0x20	,	0x00	,	0x70	,	0x00	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x00	,	0x20	,	0x00	,	0x78	,	0x00	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x04	,	0x20	,	0x04	,	0x78	,	0x04	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x06	,	0x20	,	0x06	,	0x78	,	0x06	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x07	,	0x20	,	0x07	,	0x78	,	0x07	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x07	,	0x20	,	0x07	,	0x78	,	0x07	,	0x20	,	0x00	,	0x10
   entry	0x00	,	0x10	,	0x07	,	0x20	,	0x07	,	0x78	,	0x07	,	0x20	,	0x00	,	0x90

;**********************************************************************
table13  ; 
;**********************************************************************
   tbltype1 0x10

   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x08	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x1C	,	0x00	,	0x1C	,	0x00	,	0x1C	,	0x00	,	0x00	,	0x00
   entry	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00
   entry	0x7F	,	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x00	,	0x08	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x1C	,	0x00	,	0x1C	,	0x00	,	0x1C	,	0x00	,	0x00
   entry	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E	,	0x00	,	0x3E
   entry	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00	,	0x7F	,	0x00	,	0xFF

;**********************************************************************
table14  ; 
;**********************************************************************
   tbltype1 0x10

   entry	0x01	,	0x00	,	0x02	,	0x00	,	0x01	,	0x01	,	0x02	,	0x42	,	0x04	,	0x04
   entry	0x01	,	0x08	,	0x02	,	0x00	,	0x04	,	0x00	,	0x02	,	0x02	,	0x04	,	0x04
   entry	0x00	,	0x08	,	0x02	,	0x10	,	0x04	,	0x00	,	0x08	,	0x00	,	0x04	,	0x04
   entry	0x00	,	0x08	,	0x00	,	0x10	,	0x04	,	0x20	,	0x08	,	0x00	,	0x10	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x10	,	0x01	,	0x21	,	0x08	,	0x40	,	0x10	,	0x00
   entry	0x01	,	0x00	,	0x00	,	0x00	,	0x01	,	0x21	,	0x02	,	0x42	,	0x10	,	0x80

;**********************************************************************
table15  ; 
;**********************************************************************
   tbltype1 0x10

   entry	0x05	,	0x02	,	0x02	,	0x05	,	0x05	,	0x02	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x05	,	0x02	,	0x02	,	0x05	,	0x05	,	0x02	,	0x00	,	0x00
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x05	,	0x02	,	0x02	,	0x05	,	0x05	,	0x02
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x0A	,	0x04	,	0x04	,	0x0A	,	0x0A	,	0x04
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x14	,	0x08	,	0x08	,	0x14	,	0x14	,	0x08
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x28	,	0x10	,	0x10	,	0x28	,	0x28	,	0x10
   entry	0x00	,	0x00	,	0x00	,	0x00	,	0x50	,	0x20	,	0x20	,	0x50	,	0x50	,	0x20
   entry	0x00	,	0x00	,	0x50	,	0x20	,	0x20	,	0x50	,	0x50	,	0x20	,	0x00	,	0x00
   entry	0x50	,	0x20	,	0x20	,	0x50	,	0x50	,	0x20	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x28	,	0x10	,	0x10	,	0x28	,	0x28	,	0x10	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x14	,	0x08	,	0x08	,	0x14	,	0x14	,	0x08	,	0x00	,	0x00	,	0x00	,	0x00
   entry	0x0A	,	0x04	,	0x04	,	0x0A	,	0x0A	,	0x04	,	0x00	,	0x00	,	0x00	,	0x80

;**********************************************************************
table16  ; 
;**********************************************************************
    tbltype5 0x30

;**********************************************************************
chartable  ; 
;**********************************************************************

    entrychar 0x00,0x80,0x80,0x80,0x80; 0x20 space
    entrychar 0xDF,0x80,0x80,0x80,0x80; 0x21  !
    entrychar 0x03,0x00,0x83,0x80,0x80; 0x22 "
    entrychar 0x14,0x3E,0x14,0x3E,0x94; 0x23 #
    entrychar 0x24,0x2A,0x7F,0x2A,0x92; 0x24 $
    entrychar 0x23,0x13,0x08,0x64,0xE2; 0x25 %
    entrychar 0x36,0x49,0x56,0x20,0xD0; 0x26 &
    entrychar 0x1C,0x1C,0x9C,0x80,0x80; 0x27 
    entrychar 0x1C,0x22,0xC1,0x80,0x80; 0x28 (
    entrychar 0x41,0x22,0x9C,0x80,0x80; 0x29 )
    entrychar 0x2A,0x1C,0x7F,0x1C,0xAA; 0x2A *
    entrychar 0x08,0x1C,0x88,0x80,0x80; 0x2B +
    entrychar 0x30,0xF0,0x80,0x80,0x80; 0x2C ,
    entrychar 0x08,0x08,0x88,0x80,0x80; 0x2D -
    entrychar 0x60,0xE0,0x80,0x80,0x80; 0x2E .
    entrychar 0x40,0x30,0x08,0x06,0x81; 0x2F /
    entrychar 0x3E,0x51,0x49,0x45,0xBE; 0x30 0
    entrychar 0x42,0x7F,0xC0,0x80,0x80; 0x31 1
    entrychar 0x42,0x61,0x51,0x49,0xC6; 0x32 2
    entrychar 0x22,0x41,0x49,0x49,0xB6; 0x33 3
    entrychar 0x18,0x14,0x12,0x7F,0x90; 0x34 4
    entrychar 0x27,0x45,0x45,0x45,0xB9; 0x35 5
    entrychar 0x3C,0x4A,0x49,0x49,0xB0; 0x36 6
    entrychar 0x01,0x71,0x09,0x05,0x83; 0x37 7
    entrychar 0x36,0x49,0x49,0x49,0xB6; 0x38 8
    entrychar 0x06,0x49,0x49,0x49,0xBE; 0x39 9
    entrychar 0x36,0xB6,0x80,0x80,0x80; 0x3A :
    entrychar 0x36,0xF6,0x80,0x80,0x80; 0x3B ;
    entrychar 0x08,0x14,0x22,0xC1,0x80; 0x3C <
    entrychar 0x14,0x14,0x14,0x94,0x80; 0x3D =
    entrychar 0x41,0x22,0x14,0x88,0x80; 0x3E >
    entrychar 0x02,0x01,0x51,0x09,0x86; 0x3F ?
    entrychar 0x3E,0x41,0x5D,0x55,0xDE; 0x40 @
    entrychar 0x7C,0x12,0x11,0x12,0xFC; 0x41 A
    entrychar 0x7F,0x49,0x49,0x49,0xB6; 0x42 B
    entrychar 0x3E,0x41,0x41,0x41,0xA2; 0x43 C
    entrychar 0x7F,0x41,0x41,0x41,0xBE; 0x44 D
    entrychar 0x7F,0x49,0x49,0x49,0xC1; 0x45 E
    entrychar 0x7F,0x09,0x09,0x09,0x81; 0x46 F
    entrychar 0x3E,0x41,0x49,0x49,0xBA; 0x47 G
    entrychar 0x7F,0x08,0x08,0x08,0xFF; 0x48 H
    entrychar 0x41,0x7F,0xC1,0x80,0x80; 0x49 I
    entrychar 0x20,0x40,0x40,0xBF,0x80; 0x4A J
    entrychar 0x7F,0x08,0x14,0x22,0xC1; 0x4B K
    entrychar 0x7F,0x40,0x40,0xC0,0x80; 0x4C L
    entrychar 0x7F,0x02,0x0C,0x02,0xFF; 0x4D M
    entrychar 0x7F,0x02,0x04,0x08,0xFF; 0x4E N
    entrychar 0x3E,0x41,0x41,0x41,0xBE; 0x4F O
    entrychar 0x7F,0x09,0x09,0x09,0x86; 0x50 P
    entrychar 0x3E,0x41,0x51,0x21,0xDE; 0x51 Q
    entrychar 0x7F,0x09,0x19,0x29,0xC6; 0x52 R
    entrychar 0x26,0x49,0x49,0x49,0xB2; 0x53 S
    entrychar 0x01,0x01,0x7F,0x01,0x81; 0x54 T
    entrychar 0x3F,0x40,0x40,0x40,0xBF; 0x55 U
    entrychar 0x1F,0x20,0x40,0x20,0x9F; 0x56 V
    entrychar 0x3F,0x40,0x38,0x40,0xBF; 0x57 W
    entrychar 0x63,0x14,0x08,0x14,0xE3; 0x58 X
    entrychar 0x07,0x08,0x70,0x08,0x87; 0x59 Y
    entrychar 0x61,0x51,0x49,0x45,0xC3; 0x5A Z
    entrychar 0x7F,0x41,0xC1,0x80,0x80; 0x5B [
    entrychar 0x01,0x06,0x08,0x30,0xC0; 0x5C \
    entrychar 0x41,0x41,0xFF,0x80,0x80; 0x5D ]
    entrychar 0x02,0x01,0x82,0x80,0x80; 0x5E ^
    entrychar 0x40,0x40,0x40,0x40,0xC0; 0x5F _
    entrychar 0x00,0x03,0x80,0x80,0x80; 0x61 a
    entrychar 0x20,0x54,0x54,0xF8,0x80; 0x61 a
    entrychar 0x7F,0x44,0x44,0xB8,0x80; 0x62 b
    entrychar 0x38,0x44,0x44,0xC4,0x80; 0x63 c
    entrychar 0x38,0x44,0x44,0xFF,0x80; 0x64 d
    entrychar 0x38,0x54,0x54,0xD8,0x80; 0x65 e
    entrychar 0x04,0x7E,0x05,0x81,0x80; 0x66 f
    entrychar 0x48,0x54,0x54,0xBC,0x80; 0x67 g
    entrychar 0x7F,0x08,0x04,0xF8,0x80; 0x68 h
    entrychar 0xFD,0x80,0x80,0x80,0x80; 0x69 i
    entrychar 0x20,0x40,0xBD,0x80,0x80; 0x6A j
    entrychar 0x7F,0x10,0x28,0xC4,0x80; 0x6B k
    entrychar 0xFF,0x80,0x80,0x80,0x80; 0x6C l
    entrychar 0x78,0x04,0x18,0x04,0xF8; 0x6D m
    entrychar 0x7C,0x08,0x04,0xF8,0x80; 0x6E n
    entrychar 0x38,0x44,0x44,0xB8,0x80; 0x6F o
    entrychar 0x7C,0x14,0x14,0x88,0x80; 0x70 p
    entrychar 0x08,0x14,0x14,0xFC,0x80; 0x71 q
    entrychar 0x7C,0x08,0x04,0x84,0x80; 0x72 r
    entrychar 0x48,0x54,0x54,0xA4,0x80; 0x73 s
    entrychar 0x04,0x7F,0x84,0x80,0x80; 0x74 t
    entrychar 0x3C,0x40,0x40,0xBC,0x80; 0x75 u
    entrychar 0x0C,0x30,0x40,0x30,0x8C; 0x76 v
    entrychar 0x3C,0x40,0x30,0x40,0xBC; 0x77 w
    entrychar 0x44,0x28,0x10,0x28,0xC4; 0x78 x
    entrychar 0x4C,0x50,0x50,0xBC,0x80; 0x79 y
    entrychar 0x44,0x64,0x54,0x4C,0xC4; 0x7A z
    entrychar 0x00,0x3E,0x3E,0x3E,0x80; 0x7B user for color icon
    entrychar 0x77,0x41,0x63,0x41,0xF7; 0x7C end of patter
    entrychar 0x14,0x22,0x14,0x22,0x94; 0x7D invalid character



;**********************************************************************
; eeprom
;**********************************************************************

	ORG	0x2100				; data EEPROM location
    DE  0x01

    DE  0x03, 0x14
    DE  0x04
    DE  " "
    DE  "You need to learn how to change this message with your TV remote!"
    DE  " "
    DE  0x05
    DE  "The instructions are at www.2DKits.com."
    DE  0x00


	END                       ; directive 'end of program'

